function res = chris_bursts2(temp)
% Processes burst inoformation (synaptic activity in Chris'es neurons in response to stimulation
% with different ISIs. Here ISIs are written explicitly in the program, so be careful and make sure
% they match whatever ISIs are given in the XLS file.

% Sep 17 2012: Revision.
% Sep 11 2014: Revision for 2014 set of data.
% Sep 13 2014: Reading extended (segmented) data. Slightly different processing and output for old-type data.
% Jan 19 2015: Revivied to make a figure. LIST variable introduced.

if(nargin<1)
    fileName = 'C:\Users\Arseny\Documents\4_Cells Classification\2014 data\FinalData (20140415) - AKh.xlsx';
    sheetName = 'BurstsFinalAKh';
    temp = xlsread(fileName, sheetName);
    res = temp;
    return
end

list = 22003;                       % If not all cells are needed

flagShowSegments = 0;               % Whether you want to see segmented data


isi = [100 150 200 250 300 10 20 30 40 50];         % ISIs go in this weird order
maxX = 300;                                         % Some fitting parameter

id = temp(:,1);
data1 = temp(:,2:11);                               % Total current for different ISIs
data2 = temp(:,12:end);                             % Currents-per-setmeng data.

if(exist('list','var'))                             % If only a subset of cells is to be processed
    g = ismember(id,list);
    id = id(g);
    data1 = data1(g,:);
    data2 = data2(g,:);
end

[isi, indIsi] = sort(isi);                          % Reordering ISIs in a more logical sequence (from shortest to longest)
data1 = data1(:,indIsi);                            % Sorting data accordingly
superInd = bsxfun(@plus,(indIsi-1)*5,(1:5)'-1)+1;   % Currents-per-segment data is a bit harder to sort, so building an index for that.
data2 = data2(:,superInd(:)');                      % And then reorganizing current-per-segment as well.

smoothx = linspace(isi(1),isi(end),50);             % A useful constant for plotting purposes

data1 = max(-data1,0);          % Invert the currents (make them positive), and also get rid of opposite-sign noise.
data2 = max(-data2,0);          % Same for per-segment data.

[n,l] = size(data1);
plotN = max(1,floor(sqrt(n)/1.3)); % Layout for the multiple currents plot
plotM = ceil(n/plotN);

if(flagShowSegments)
    for(q=1:n)
        if(mod(q-1,plotN*plotM)==0)
            figure('Color','white');
        end
        subplot(plotN,plotM,mod(q-1,plotN*plotM)+1);
        hold on;

        squareData = reshape(data2(q,:),5,[]);
        xlim([1 5.6]);
        for(iIsi=1:length(isi))
            plot(squareData(:,iIsi),'.-','Color',mycolor(iIsi));
            text(5.2,squareData(5,iIsi),num2str(isi(iIsi)),'FontSize',6,'Color',mycolor(iIsi)*0.8);
        end
        set(gca,'FontSize',6);
        title(id(q),'FontSize',8);

        hold off;   
        drawnow;
        % if(q==plotN*plotM); return; end;            % Debugger emergency break    
    end
end


[f,s] = buildAModel('xexp',maxX);

fprintf('\n');
dispf({'id','Best ISI','Max Q','Amplification','t decay'},'%12s');
for(q=1:n)
    if(mod(q-1,plotN*plotM)==0)
        figure('Color','white');
    end
    subplot(plotN,plotM,mod(q-1,plotN*plotM)+1);
    hold on;
    
    [c2,gof2] = fit(isi',data1(q,:)',f,s);
    fmaxx = fminbnd(@(x)(-c2(x)),isi(1),maxX);  % Using anonymous function to find curve maximum
    maxy = min(c2(fmaxx),max(data1(q,:)));
    
    plot(smoothx,c2(smoothx),'r-');
    plot(isi',data1(q,:)','b.');    
    plot(fmaxx,maxy,'ko');
    set(gca,'XLim',[0 maxX],'FontSize',6);
    title(id(q));
    hold off;
    drawnow;
    %fprintf('%8d\t%5.1f\t%5.1f\t%5.1f\t%5.1f\n',[id(q) c2.c c2.e c2(fmaxx) fmaxx]);
    dispf([id(q) fmaxx maxy c2(fmaxx)/min(c2(isi(end)),data1(q,end)) c2.c],'%12.3f');
    
    % if(q==plotN*plotM); return; end;            % Debugger emergency break
end

end



function [f,s] = buildAModel(prompt,maxX)
% Model library to fit the points

switch(prompt)
    case 'exp'
        f = fittype('exp(-(x-a)/b)*c+d','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [-maxX   1   0   0],...
                       'Upper',     [maxX    maxX Inf Inf],...
                       'Startpoint',[0      10  1   1]);
    case 'sigmoid'
        f = fittype('c/(1+exp((x-a)/b)+d)','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [-maxX   1   0   0],...
                       'Upper',     [maxX    maxX Inf Inf],...
                       'Startpoint',[0      10  1   1]);
   case 'xexp'
        f = fittype('(x-a)*exp(-(x-b)/c)*d+e','independent','x');
        s = fitoptions('Method','NonlinearLeastSquares',...
                       'Lower',     [-maxX   -maxX   1   0   0],...
                       'Upper',     [maxX    maxX    maxX Inf Inf],...
                       'Startpoint',[0      0      100  1   1]);
end

end


function m = mycolor(k)
% Pre-coocked colorbrewer colors
table = [   1.0000    0.8941    0.5451
    0.9961    0.8510    0.4627
    0.9961    0.6980    0.2980
    0.9961    0.6314    0.2627
    0.9922    0.5529    0.2353
    0.9882    0.3059    0.1647
    0.8902    0.1020    0.1098
    0.8235    0.0275    0.1294
    0.7412         0    0.1490
    0.5020         0    0.1490];
m = table(k,:);
end